// Content Script - Runs on web pages
console.log('IAR Extension: Content script loaded');

// Create floating button for quick access
let floatingButton = null;
let selectedText = '';

// Initialize
function init() {
  createFloatingButton();
  setupTextSelection();
  setupInstagramIntegration();
}

// Create floating button
function createFloatingButton() {
  floatingButton = document.createElement('div');
  floatingButton.id = 'iar-floating-button';
  floatingButton.innerHTML = `
    <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
      <path d="M13 2L3 14H12L11 22L21 10H12L13 2Z" stroke="white" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
    </svg>
  `;
  floatingButton.style.display = 'none';
  document.body.appendChild(floatingButton);

  floatingButton.addEventListener('click', handleFloatingButtonClick);
}

// Setup text selection listener
function setupTextSelection() {
  document.addEventListener('mouseup', handleTextSelection);
  document.addEventListener('touchend', handleTextSelection);
}

// Handle text selection
function handleTextSelection(e) {
  setTimeout(() => {
    const selection = window.getSelection();
    const text = selection.toString().trim();

    if (text && text.length > 10) {
      selectedText = text;
      showFloatingButton(e);
    } else {
      hideFloatingButton();
    }
  }, 10);
}

// Show floating button near selection
function showFloatingButton(e) {
  if (!floatingButton) return;

  const x = e.pageX || e.changedTouches?.[0]?.pageX || 0;
  const y = e.pageY || e.changedTouches?.[0]?.pageY || 0;

  floatingButton.style.left = `${x + 10}px`;
  floatingButton.style.top = `${y - 50}px`;
  floatingButton.style.display = 'flex';
}

// Hide floating button
function hideFloatingButton() {
  if (floatingButton) {
    floatingButton.style.display = 'none';
  }
}

// Handle floating button click
function handleFloatingButtonClick() {
  if (selectedText) {
    chrome.runtime.sendMessage({
      action: 'analyzeText',
      text: selectedText
    });
    hideFloatingButton();
  }
}

// Instagram-specific integration
function setupInstagramIntegration() {
  if (!window.location.hostname.includes('instagram.com')) return;

  console.log('IAR Extension: Instagram integration active');

  // Add analyze button to bio
  addAnalyzeToBio();

  // Add analyze button to posts
  observeInstagramContent();
}

// Add analyze button to Instagram bio
function addAnalyzeToBio() {
  const bioSelector = 'header section div[style*="word-wrap"]';
  
  const checkBio = setInterval(() => {
    const bioElement = document.querySelector(bioSelector);
    
    if (bioElement && !bioElement.querySelector('.iar-analyze-btn')) {
      const analyzeBtn = createAnalyzeButton('Analisar Bio com IAR');
      analyzeBtn.addEventListener('click', () => {
        const bioText = bioElement.textContent;
        chrome.runtime.sendMessage({
          action: 'analyzeText',
          text: bioText,
          type: 'bio'
        });
      });
      
      bioElement.parentElement.appendChild(analyzeBtn);
      clearInterval(checkBio);
    }
  }, 1000);

  // Clear after 10 seconds if not found
  setTimeout(() => clearInterval(checkBio), 10000);
}

// Observe Instagram content changes
function observeInstagramContent() {
  const observer = new MutationObserver((mutations) => {
    // Add analyze buttons to new posts
    addAnalyzeToVisiblePosts();
  });

  observer.observe(document.body, {
    childList: true,
    subtree: true
  });
}

// Add analyze buttons to visible posts
function addAnalyzeToVisiblePosts() {
  const posts = document.querySelectorAll('article');
  
  posts.forEach(post => {
    if (post.querySelector('.iar-analyze-btn')) return;

    const captionElement = post.querySelector('span[style*="word-wrap"]');
    if (!captionElement) return;

    const analyzeBtn = createAnalyzeButton('Analisar com IAR', true);
    analyzeBtn.addEventListener('click', () => {
      const captionText = captionElement.textContent;
      chrome.runtime.sendMessage({
        action: 'analyzeText',
        text: captionText,
        type: 'caption'
      });
    });

    // Insert button near caption
    const container = captionElement.closest('div[style*="padding"]');
    if (container) {
      container.appendChild(analyzeBtn);
    }
  });
}

// Create analyze button
function createAnalyzeButton(text, small = false) {
  const button = document.createElement('button');
  button.className = 'iar-analyze-btn';
  button.textContent = text;
  button.style.cssText = small ? 'font-size: 12px; padding: 4px 8px;' : '';
  return button;
}

// Context menu for right-click
document.addEventListener('contextmenu', (e) => {
  const selection = window.getSelection().toString().trim();
  if (selection) {
    selectedText = selection;
  }
});

// Listen for messages from popup
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'getSelectedText') {
    sendResponse({ text: selectedText });
  }
});

// Initialize when DOM is ready
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', init);
} else {
  init();
}

// Cleanup on unload
window.addEventListener('beforeunload', () => {
  if (floatingButton) {
    floatingButton.remove();
  }
});
