// API Configuration
const API_URL = 'https://mindai-hub.onrender.com/api';

// DOM Elements - License
const authSection = document.getElementById('authSection');
const mainContent = document.getElementById('mainContent');
const licenseKeyInput = document.getElementById('licenseKeyInput');
const activateLicenseBtn = document.getElementById('activateLicenseBtn');
const licenseError = document.getElementById('licenseError');
const testModeBtn = document.getElementById('testModeBtn');
const editProfileBtn = document.getElementById('editProfileBtn');

// DOM Elements - Tabs
const tabs = document.querySelectorAll('.tab');
const tabContents = document.querySelectorAll('.tab-content');

// DOM Elements - Central de Ideias
const gerarIdeiasBtn = document.getElementById('gerarIdeiasBtn');
const ideiasResults = document.getElementById('ideiasResults');
const ideiasContent = document.getElementById('ideiasContent');
const copyIdeiasBtn = document.getElementById('copyIdeias');

// DOM Elements - Conteúdos Neurais
const temaConteudoInput = document.getElementById('temaConteudo');
const formatoConteudoSelect = document.getElementById('formatoConteudo');
const gerarConteudoBtn = document.getElementById('gerarConteudoBtn');
const conteudoResults = document.getElementById('conteudoResults');
const conteudoContent = document.getElementById('conteudoContent');
const copyConteudoBtn = document.getElementById('copyConteudo');

// DOM Elements - Neuro Respostas
const tipoRespostaSelect = document.getElementById('tipoResposta');
const mensagemLeadInput = document.getElementById('mensagemLead');
const gerarRespostasBtn = document.getElementById('gerarRespostasBtn');
const respostasResults = document.getElementById('respostasResults');
const resposta1El = document.getElementById('resposta1');
const resposta2El = document.getElementById('resposta2');
const resposta3El = document.getElementById('resposta3');
const copyRespostaBtns = document.querySelectorAll('.copy-resposta');

// Initialize
document.addEventListener('DOMContentLoaded', async () => {
  setupEventListeners();
  await checkLicenseStatus();
});

function setupEventListeners() {
  // License
  activateLicenseBtn?.addEventListener('click', handleActivateLicense);
  testModeBtn?.addEventListener('click', handleTestMode);
  editProfileBtn?.addEventListener('click', handleEditProfile);

  // Tabs
  tabs.forEach(tab => {
    tab.addEventListener('click', () => switchTab(tab.dataset.tab));
  });

  // Central de Ideias
  gerarIdeiasBtn?.addEventListener('click', handleGerarIdeias);
  copyIdeiasBtn?.addEventListener('click', () => copyToClipboard(ideiasContent.textContent));

  // Conteúdos Neurais
  gerarConteudoBtn?.addEventListener('click', handleGerarConteudo);
  copyConteudoBtn?.addEventListener('click', () => copyToClipboard(conteudoContent.textContent));

  // Neuro Respostas
  gerarRespostasBtn?.addEventListener('click', handleGerarRespostas);
  copyRespostaBtns.forEach(btn => {
    btn.addEventListener('click', (e) => {
      const index = e.target.dataset.index;
      const respostaEl = document.getElementById(`resposta${index}`);
      if (respostaEl) copyToClipboard(respostaEl.textContent);
    });
  });
}

// License Management
async function checkLicenseStatus() {
  const data = await chrome.storage.local.get(['mindai_license_status']);
  
  if (data.mindai_license_status === 'active') {
    showMainContent();
  } else {
    showAuthSection();
  }
}

async function handleActivateLicense() {
  const licenseKey = licenseKeyInput.value.trim().toUpperCase();
  
  if (!licenseKey) {
    showLicenseError('Digite uma chave de licença');
    return;
  }

  if (!licenseKey.startsWith('MINDAI-')) {
    showLicenseError('Formato de chave inválido');
    return;
  }

  activateLicenseBtn.disabled = true;
  activateLicenseBtn.textContent = 'Ativando...';

  try {
    const response = await fetch(`${API_URL}/mindai/license/validate`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ license_key: licenseKey })
    });

    const data = await response.json();

    if (data.valid) {
      await chrome.storage.local.set({
        mindai_license_key: licenseKey,
        mindai_license_status: 'active',
        mindai_license_plan: data.plan
      });
      
      showMainContent();
    } else {
      showLicenseError(data.error || 'Chave inválida');
    }
  } catch (err) {
    showLicenseError('Erro ao conectar. Verifique sua conexão.');
  } finally {
    activateLicenseBtn.disabled = false;
    activateLicenseBtn.textContent = 'Ativar Licença';
  }
}

async function handleTestMode(e) {
  e?.preventDefault();
  e?.stopPropagation();
  
  const mockProfile = {
    nicho: 'Infoempreendedorismo',
    promessa: 'Ajudo infoempreendedores a criar conteúdo que converte',
    transformacao: 'De invisível para autoridade no nicho',
    tom_de_voz: 'Motivador e direto',
    persona: 'Infoempreendedores iniciantes que lutam para criar conteúdo'
  };
  
  await chrome.storage.local.set({
    mindai_license_status: 'active',
    mindai_license_plan: 'test',
    mindai_user_profile: mockProfile
  });
  
  showMainContent();
}

function handleEditProfile() {
  window.location.href = 'diagnostic.html';
}

function showAuthSection() {
  authSection.style.display = 'block';
  mainContent.style.display = 'none';
  editProfileBtn.style.display = 'none';
}

function showMainContent() {
  authSection.style.display = 'none';
  mainContent.style.display = 'block';
  editProfileBtn.style.display = 'flex';
  loadUserProfile();
}

async function loadUserProfile() {
  const data = await chrome.storage.local.get(['mindai_user_profile', 'mindai_license_plan']);
  
  if (data.mindai_user_profile) {
    const profile = data.mindai_user_profile;
    document.getElementById('userName').textContent = profile.nicho || 'Infoempreendedor';
    document.getElementById('userInitial').textContent = (profile.nicho || 'I')[0].toUpperCase();
  }
  
  if (data.mindai_license_plan) {
    document.getElementById('userPlan').textContent = `Plano ${data.mindai_license_plan}`;
  }
}

function showLicenseError(message) {
  licenseError.textContent = message;
  licenseError.style.display = 'block';
  setTimeout(() => {
    licenseError.style.display = 'none';
  }, 5000);
}

// Tab Switching
function switchTab(tabName) {
  tabs.forEach(tab => tab.classList.remove('active'));
  tabContents.forEach(content => content.classList.remove('active'));
  
  const activeTab = document.querySelector(`[data-tab="${tabName}"]`);
  const activeContent = document.getElementById(`${tabName}Tab`);
  
  activeTab?.classList.add('active');
  activeContent?.classList.add('active');
}

// Central de Ideias
async function handleGerarIdeias() {
  const profile = await getUserProfile();
  
  if (!profile) {
    alert('Complete seu perfil primeiro');
    handleEditProfile();
    return;
  }

  gerarIdeiasBtn.disabled = true;
  gerarIdeiasBtn.textContent = '⏳ Gerando ideias...';
  ideiasResults.style.display = 'none';

  try {
    const response = await fetch(`${API_URL}/generate-content-ideas`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ userProfile: profile })
    });

    const data = await response.json();
    
    ideiasContent.textContent = data.ideas || 'Erro ao gerar ideias';
    ideiasResults.style.display = 'block';
  } catch (err) {
    alert('Erro ao conectar com API');
  } finally {
    gerarIdeiasBtn.disabled = false;
    gerarIdeiasBtn.textContent = '✨ Gerar 5 Ideias';
  }
}

// Conteúdos Neurais
async function handleGerarConteudo() {
  const profile = await getUserProfile();
  
  if (!profile) {
    alert('Complete seu perfil primeiro');
    handleEditProfile();
    return;
  }

  const tema = temaConteudoInput.value.trim();
  const formato = formatoConteudoSelect.value;

  if (!tema || !formato) {
    alert('Preencha o tema e escolha o formato');
    return;
  }

  gerarConteudoBtn.disabled = true;
  gerarConteudoBtn.textContent = '⏳ Gerando conteúdo...';
  conteudoResults.style.display = 'none';

  try {
    const response = await fetch(`${API_URL}/generate-neural-content`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ 
        userProfile: profile,
        tema,
        formato
      })
    });

    const data = await response.json();
    
    conteudoContent.textContent = data.content || 'Erro ao gerar conteúdo';
    conteudoResults.style.display = 'block';
  } catch (err) {
    alert('Erro ao conectar com API');
  } finally {
    gerarConteudoBtn.disabled = false;
    gerarConteudoBtn.textContent = '✨ Gerar Conteúdo Completo';
  }
}

// Neuro Respostas
async function handleGerarRespostas() {
  const profile = await getUserProfile();
  
  if (!profile) {
    alert('Complete seu perfil primeiro');
    handleEditProfile();
    return;
  }

  const tipo = tipoRespostaSelect.value;
  const mensagem = mensagemLeadInput.value.trim();

  if (!tipo || !mensagem) {
    alert('Escolha o tipo e cole a mensagem');
    return;
  }

  gerarRespostasBtn.disabled = true;
  gerarRespostasBtn.textContent = '⏳ Gerando respostas...';
  respostasResults.style.display = 'none';

  try {
    const response = await fetch(`${API_URL}/generate-neuro-responses`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ 
        userProfile: profile,
        mensagem,
        tipo
      })
    });

    const data = await response.json();
    
    if (data.responses && data.responses.length >= 3) {
      resposta1El.textContent = data.responses[0];
      resposta2El.textContent = data.responses[1];
      resposta3El.textContent = data.responses[2];
      respostasResults.style.display = 'block';
    } else {
      alert('Erro ao gerar respostas');
    }
  } catch (err) {
    alert('Erro ao conectar com API');
  } finally {
    gerarRespostasBtn.disabled = false;
    gerarRespostasBtn.textContent = '💬 Gerar 3 Versões';
  }
}

// Utility Functions
async function getUserProfile() {
  const data = await chrome.storage.local.get(['mindai_user_profile']);
  return data.mindai_user_profile || null;
}

function copyToClipboard(text) {
  navigator.clipboard.writeText(text).then(() => {
    const btn = event.target;
    const originalText = btn.textContent;
    btn.textContent = '✓ Copiado!';
    setTimeout(() => {
      btn.textContent = originalText;
    }, 2000);
  });
}
